' sap-script-scope-check.vbs - Checks inspection scope and handles navigation flow
' Runs after sap-script-rework-and-tool-type and before inspection details

On Error Resume Next

' Set scripting mode to true
Set WSHShell = CreateObject("WScript.Shell")

' Disable SAP GUI scripting warnings in registry
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' Declare log file object and path
Dim logFile, logFilePath

' Get command line arguments first
materialNumber = WScript.Arguments.Item(0)  ' Material number for ZPACK

' Initialize log file if path is provided
If WScript.Arguments.Count > 1 Then
    logFilePath = WScript.Arguments.Item(1)
    ' Try to open the log file in append mode (8)
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Log script start
        LogToFile "========================================="
        LogToFile "Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Material Number: " & materialNumber
        LogToFile "========================================="
    End If
End If

' Function to log message to file
Sub LogToFile(message)
    On Error Resume Next
    If Not logFile Is Nothing Then
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Combined logging function that writes to both console and file
Sub LogMessage(message)
    WScript.Echo message
    LogToFile message
End Sub

' Initialize SAP connection
If Not IsObject(application) Then
   Set SapGuiAuto = GetObject("SAPGUI")
   Set application = SapGuiAuto.GetScriptingEngine
End If
If Not IsObject(connection) Then
   Set connection = application.Children(0)
End If
If Not IsObject(session) Then
   Set session = connection.Children(0)
End If
If IsObject(WScript) Then
   WScript.ConnectObject session, "on"
   WScript.ConnectObject application, "on"
End If

' Handle security dialog
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    Do While timeout < 10
        If shell.AppActivate("SAP GUI Security") Then
            WScript.Sleep 500
            shell.SendKeys " "  ' Press space to check "Remember my decision"
            WScript.Sleep 200
            shell.SendKeys "{ENTER}"  ' Press Enter to click Allow
            dialogFound = True
            Exit Do
        End If
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' Function to show popup modal to user - handled in JavaScript now
Sub ShowOutOfScopePopup()
    ' Popup is now handled in JavaScript - this function is kept for compatibility
    ' but does nothing since the popup is shown in the UI
End Sub

' Main scope checking logic
Sub CheckInspectionScope()
    LogMessage "DEBUG: Starting scope check for material: " & materialNumber
    
    ' Call HandleSecurityDialog before starting operations
    HandleSecurityDialog()
    
    ' Navigate to ZPACK transaction
    session.findById("wnd[0]").maximize
    session.findById("wnd[0]/tbar[0]/okcd").text = "zpack"
    session.findById("wnd[0]").sendVKey 0
    
    ' Enter material number and execute search
    session.findById("wnd[0]/usr/ctxtS_MATNR-LOW").text = materialNumber
    session.findById("wnd[0]/tbar[1]/btn[8]").press
    
    ' Get the text field content
    On Error Resume Next
    Set textField = session.findById("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/txtWA_OUTPUT_9100-OPERCOM")
    
    If Err.Number <> 0 Then
        LogMessage "ERROR: Could not find text field: " & Err.Description
        ' Navigate back and exit with error
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        WScript.Echo "ERROR-FIELD-NOT-FOUND|Field not found"
        Exit Sub
    End If
    
    ' Focus on the text field to ensure we can read it
    textField.setFocus
    WScript.Sleep 1000
    
    ' Get the text content with proper handling of empty values
    Dim scopeText, originalText
    originalText = ""
    On Error Resume Next
    originalText = textField.Text
    If Err.Number <> 0 Then
        originalText = ""
        Err.Clear
    End If
    On Error Goto 0
    
    ' Ensure we have a string value for comparison
    If IsNull(originalText) Then originalText = ""
    If originalText = Empty Then originalText = ""
    
    scopeText = Trim(UCase(originalText))
    
    LogMessage "DEBUG: Found scope text: '" & scopeText & "'"
    LogMessage "DEBUG: Raw text field content: '" & originalText & "'"
    LogMessage "DEBUG: Text field length: " & Len(originalText)
    LogMessage "DEBUG: Looking for 'OUT OF SCOPE FOR INSPECTION' pattern..."
    LogMessage "DEBUG: Looking for 'OUT OF SCOPE' pattern..."
    LogMessage "DEBUG: Looking for 'IN SCOPE FOR INSPECTION' pattern..."
    
    ' Additional debugging - show what we're actually comparing
    LogMessage "DEBUG: InStr result for 'OUT OF SCOPE FOR INSPECTION': " & InStr(scopeText, "OUT OF SCOPE FOR INSPECTION")
    LogMessage "DEBUG: InStr result for 'OUT OF SCOPE': " & InStr(scopeText, "OUT OF SCOPE")
    LogMessage "DEBUG: InStr result for 'IN SCOPE FOR INSPECTION': " & InStr(scopeText, "IN SCOPE FOR INSPECTION")
    
    ' Check the scope text content
    If InStr(scopeText, "OUT OF SCOPE FOR INSPECTION") > 0 Or InStr(scopeText, "OUT OF SCOPE") > 0 Then
        LogMessage "INFO: Tool is out of scope for inspection"
          ' Navigate back to main screen first
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        
        LogMessage "DEBUG: Returning OUT-OF-SCOPE with text: '" & originalText & "'"
        ' Return status to indicate out of scope and return to field service
        ' Include the actual text content
        WScript.Echo "OUT-OF-SCOPE-RETURN-TO-FIELD-SERVICE|" & originalText
        
    ElseIf InStr(scopeText, "IN SCOPE FOR INSPECTION") > 0 Then
        LogMessage "INFO: Tool is in scope for inspection - proceeding to inspection details"
        
        ' Navigate back to prepare for next page
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        
        LogMessage "DEBUG: Returning IN-SCOPE with text: '" & originalText & "'"
        ' Return status to proceed to inspection details
        ' Include the actual text content
        WScript.Echo "IN-SCOPE-PROCEED-TO-INSPECTION|" & originalText
        
    Else
        LogMessage "INFO: Scope text is something else ('" & scopeText & "') - proceeding to inspection details"
        
        ' Navigate back to prepare for next page
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        session.findById("wnd[0]/tbar[0]/btn[3]").press
        
        LogMessage "DEBUG: Returning OTHER-TEXT with text: '" & originalText & "'"
        ' Return status to proceed to inspection details (default behavior)
        ' Include the actual text content
        WScript.Echo "OTHER-TEXT-PROCEED-TO-INSPECTION|" & originalText
    End If
End Sub

' Execute the main logic
CheckInspectionScope()

' Close the log file at the end of script
Sub CloseLogFile()
    On Error Resume Next
    If Not logFile Is Nothing Then
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        logFile.Close
        Set logFile = Nothing
    End If
End Sub

CloseLogFile()
